//////////
//
//	File:		ComApplication.c
//
//	Contains:	Application-specific code for QuickTime movies demo.
//				This file is used for BOTH MacOS and Windows.
//
//	Written by:	Tim Monroe
//
//	Copyright:	 1994-1998 by Apple Computer, Inc., all rights reserved.
//
//	Change History (most recent first):
//
//	   <11>	 	10/23/97	rtm		moved InitializeQTVR to InitApplication, TerminateQTVR to StopApplication
//	   <10>	 	10/13/97	rtm		reworked HandleApplicationMenu to use menu identifiers
//	   <9>	 	09/11/97	rtm		merged MacApplication.c and WinApplication.c into ComApplication.c
//	   <8>	 	08/21/97	rtm		first file for Windows; based on MacApplication.c for Mac sample code
//	   <7>	 	06/04/97	rtm		removed call to QTVRUtils_IsQTVRMovie in InitApplicationWindowObject
//	   <6>	 	02/06/97	rtm		fixed window resizing code
//	   <5>	 	12/11/96	rtm		begun QuickTime movie work
//	   <4>	 	12/05/96	rtm		added hooks into MacFramework.c: StopApplication, InitApplicationWindowObject
//	   <3>	 	12/02/96	rtm		added cursor updating to DoIdle
//	   <2>	 	11/27/96	rtm		conversion to personal coding style; added preliminary QTVR support
//	   <1>	 	12/21/94	khs		first file
//	   
//////////

//
// TO DO:
//

// header files
#include "ComApplication.h"

// header file for the movie functions.
#include "VRMovies.h"

// global variables for Macintosh code
#if TARGET_OS_MAC
Str255					gAppName = "\pVRMovies";				// the name of this application
#endif

// global variables for Windows code
#if TARGET_OS_WIN32
extern HWND				ghWnd;									// the MDI frame window; this window has the menu bar
extern int				gNumWindowsOpen;
extern LPSTR			gCmdLine;
#endif

long					gMaxMilliSecToUse = 0L;		
Boolean					gQTVRMgrIsPresent = false;				// is the QuickTime VR Manager available?		
long					gQTVRMgrVersion = 0L;					// the version of the QuickTime VR Manager	


//////////
//
// InitApplication
// Do any application-specific initialization.
//
// The theStartPhase parameter determines which "phase" of application start-up is executed,
// before the MDI frame window is created or after. This distinction is relevant only on
// Windows, so on MacOS, you should always use kInitAppPhase_BothPhases.
//
//////////

void InitApplication (UInt32 theStartPhase)
{
	// do any start-up activities that should occur before the MDI frame window is created
	if (theStartPhase & kInitAppPhase_BeforeCreateFrameWindow) {

		// make sure that the QuickTime VR Manager is available in the present operating environment;
		// if it is, get its version and (if necessary) initialize it
		if (QTVRUtils_IsQTVRMgrInstalled()) {
			gQTVRMgrIsPresent = true;
			gQTVRMgrVersion = QTVRUtils_GetQTVRVersion();		// get the version of the QuickTime VR Manager
		
#if TARGET_OS_WIN32
			// initialize the QuickTime VR Manager
			InitializeQTVR();									
#endif
		}
		
	}	// end of kInitAppPhase_BeforeCreateFrameWindow

	// do any start-up activities that should occur after the MDI frame window is created
	if (theStartPhase & kInitAppPhase_AfterCreateFrameWindow) {
		
#if TARGET_OS_WIN32
		// on Windows, open as movie documents any files specified on the command line
		DoOpenCommandLineMovies(gCmdLine);									
#endif

	}	// end of kInitAppPhase_AfterCreateFrameWindow
}


//////////
//
// StopApplication
// Do any application-specific shut-down.
//
// The theStopPhase parameter determines which "phase" of application shut-down is executed,
// before any open movie windows are destroyed or after.
//
//////////

void StopApplication (UInt32 theStopPhase)
{
	// do any shut-down activities that should occur after the movie windows are destroyed
	if (theStopPhase & kStopAppPhase_AfterDestroyWindows) {
	
#if TARGET_OS_WIN32
		// terminate QuickTime VR Manager
		if (gQTVRMgrIsPresent)
			TerminateQTVR();
#endif
			
	}	// end of kStopAppPhase_AfterDestroyWindows
}


//////////
//
// DoIdle
// Do any processing that can/should occur at idle time.
//
//////////

void DoIdle (WindowReference theWindow)
{
	WindowObject 		myWindowObject = NULL;
	GrafPtr 			mySavedPort;
	
	GetPort(&mySavedPort);
	MacSetPort(GetPortFromWindowReference(theWindow));
	
	myWindowObject = GetWindowObjectFromWindow(theWindow);
	if (myWindowObject != NULL) {
		MovieController		myMC = NULL;
	
		myMC = (**myWindowObject).fController;
		if (myMC != NULL) {

#if TARGET_OS_MAC
			// restore the cursor to the arrow
			// if it's outside the front movie window or outside the window's visible region
			if (theWindow == GetFrontMovieWindow()) {
				Rect	myRect;
				Point	myPoint;
				
				GetMouse(&myPoint);
				MCGetControllerBoundsRect(myMC, &myRect);
				if (!MacPtInRect(myPoint, &myRect) || !PtInRgn(myPoint, GetPortFromWindowReference(theWindow)->visRgn))
					MacSetCursor(&qd.arrow);
			}
#endif	// TARGET_OS_MAC
		}
	}

	MacSetPort(mySavedPort);
}


//////////
//
// DoUpdateWindow
// Update the specified window.
//
//////////

void DoUpdateWindow (WindowReference theWindow, Rect *theRefreshArea)
{
	GrafPtr 			mySavedPort;
	
	GetPort(&mySavedPort);
	MacSetPort(GetPortFromWindowReference(theWindow));
	
	BeginUpdate(GetPortFromWindowReference(theWindow));
	
	// draw the movie controller and its movie
	MCDoAction(GetMCFromWindow(theWindow), mcActionDraw, theWindow);
	
	EndUpdate(GetPortFromWindowReference(theWindow));
	MacSetPort(mySavedPort);
}


//////////
//
// HandleContentClick
// Handle mouse button clicks in the specified window.
//
//////////

void HandleContentClick (WindowReference theWindow, EventRecord *theEvent)
{
	GrafPtr 			mySavedPort;
	
	GetPort(&mySavedPort);
	MacSetPort(GetPortFromWindowReference(theWindow));
	
	// @@@INSERT APPLICATION-SPECIFIC CONTENT CLICKING FUNCTIONALITY HERE

	MacSetPort(mySavedPort);
}


//////////
//
// HandleApplicationKeyPress
// Handle application-specific key presses.
// Returns true if the key press was handled, false otherwise.
//
//////////

Boolean HandleApplicationKeyPress (char theCharCode)
{
	Boolean		isHandled = true;
	
	switch (theCharCode) {
	
		// @@@HANDLE APPLICATION-SPECIFIC KEY PRESSES HERE

		default:
			isHandled = false;
			break;
	}

	return(isHandled);
}


#if TARGET_OS_MAC
//////////
//
// CreateMovieWindow
// Create a window to display a movie in.
//
//////////

WindowRef CreateMovieWindow (Rect *theRect, Str255 theTitle)
{
	WindowRef			myWindow;
		
	myWindow = NewCWindow(NULL, theRect, theTitle, false, noGrowDocProc, (WindowPtr)-1L, true, 0);
	return(myWindow);
}
#endif


//////////
//
// HandleApplicationMenu
// Handle selections in the application's menus.
//
// The theMenuItem parameter is a UInt16 version of the Windows "menu item identifier". 
// When called from Windows, theMenuItem is simply the menu item identifier passed to the window proc.
// When called from MacOS, theMenuItem is constructed like this:
// 	high-order 8 bits == the Macintosh menu ID (1 thru 256)
// 	low-order 8 bits == the Macintosh menu item (sequential from 1 to ordinal of last menu item in menu)
// In this way, we can simplify the menu-handling code. There are, however, some limitations,
// mainly that the menu item identifiers on Windows must be derived from the Mac values. 
//
//////////

void HandleApplicationMenu (UInt16 theMenuItem)
{
	WindowObject		myWindowObject = NULL;
	MovieController 	myMC = NULL;
	
	myWindowObject = GetWindowObjectFromFrontWindow();
	if (myWindowObject != NULL)
		myMC = (**myWindowObject).fController;
	
	// make sure we have a valid movie controller
	if (myMC == NULL)
		return;
	
	switch (theMenuItem) {
	
		case IDM_LOAD_MOVIE:
			VRMoov_DumpEmbeddedMovie(myWindowObject);
			VRMoov_GetEmbeddedMovie(myWindowObject);
			break;

		case IDM_SET_CENTER: {
			QTVRFloatPoint	myPoint;
			
			VRMoov_GetEmbeddedMovieCenter(myWindowObject, &myPoint);
			myPoint.x += QTVRUtils_DegreesToRadians(10.0);
			VRMoov_SetEmbeddedMovieCenter(myWindowObject, &myPoint);
			break;
		}

		case IDM_SET_WIDTH:
			VRMoov_SetEmbeddedMovieWidth(myWindowObject, QTVRUtils_DegreesToRadians(20.0));
			break;

		case IDM_SET_SCALE:
			VRMoov_SetEmbeddedMovieScale(myWindowObject, 2.0);
			break;

		case IDM_SET_RECTANGLE: {
			Rect	myRect;
			
			myRect.right = 120;
			myRect.bottom = 120;
			VRMoov_SetEmbeddedMovieRect(myWindowObject, &myRect);
			break;
		}
		
		case IDM_SET_CHROMA:
			VRMoov_SetChromaColor(myWindowObject);
			break;

		case IDM_SPIN_MOVIE: {
			QTVRFloatPoint	myPoint;
			short			myIndex;
			
			for (myIndex = 0; myIndex <= 360; myIndex = myIndex + 10) {
				myPoint.x = QTVRUtils_DegreesToRadians(myIndex);
				myPoint.y = 0.0;
				VRMoov_SetEmbeddedMovieCenter(myWindowObject, &myPoint);
				QTVRSetPanAngle((**myWindowObject).fInstance, myPoint.x);
				QTVRSetTiltAngle((**myWindowObject).fInstance, myPoint.y);
				
				QTVRUpdate((**myWindowObject).fInstance, kQTVRCurrentMode);
			}
			
			break;
		}

		// need to reload movie after these are selected
		
		case IDM_USE_BUFFER: {
			ApplicationDataHdl	myAppData;
			
			myAppData = (ApplicationDataHdl)GetAppDataFromWindowObject(myWindowObject);
			if (myAppData == NULL) 
				break;
			(**myAppData).fUseMovieGWorld = !(**myAppData).fUseMovieGWorld;
			break;
		}
		
		case IDM_USE_WND_CENTER: {
			ApplicationDataHdl	myAppData;
			
			myAppData = (ApplicationDataHdl)GetAppDataFromWindowObject(myWindowObject);
			if (myAppData == NULL) 
				break;
			(**myAppData).fUseMovieCenter = !(**myAppData).fUseMovieCenter;
			break;
		}
		
		case IDM_COMPOSITE: {
			ApplicationDataHdl	myAppData;
			
			myAppData = (ApplicationDataHdl)GetAppDataFromWindowObject(myWindowObject);
			if (myAppData == NULL) 
				break;
			(**myAppData).fCompositeMovie = !(**myAppData).fCompositeMovie;
			break;
		}

		default:
			break;
	}	// switch (theMenuItem)
}


//////////
//
// AdjustApplicationMenus
// Adjust state of items in the application's menus.
//
// Currently, the Mac application has only one app-specific menu ("Test"); you could change that.
//
//////////

void AdjustApplicationMenus (WindowReference theWindow, MenuReference theMenu)
{
	ApplicationDataHdl	myAppData;	
	MenuReference		myMenu;
	
#if TARGET_OS_WIN32
	myMenu = theMenu;
#elif TARGET_OS_MAC
	myMenu = GetMenuHandle(kTestMenu);
#endif
	
	myAppData = (ApplicationDataHdl)GetAppDataFromFrontWindow();

	// we don't allow creating new files here...
#if TARGET_OS_MAC
	SetMenuItemState(GetMenuHandle(mFile), iNew, kDisableMenuItem);
#endif

	if (myAppData != NULL) {
		SetMenuItemState(myMenu, IDM_LOAD_MOVIE, kEnableMenuItem);
		SetMenuItemState(myMenu, IDM_SET_CENTER, kEnableMenuItem);
		SetMenuItemState(myMenu, IDM_SET_WIDTH, kEnableMenuItem);
		SetMenuItemState(myMenu, IDM_SET_SCALE, kEnableMenuItem);
		SetMenuItemState(myMenu, IDM_SET_RECTANGLE, kEnableMenuItem);
		SetMenuItemState(myMenu, IDM_SET_CHROMA, kEnableMenuItem);
		SetMenuItemState(myMenu, IDM_SPIN_MOVIE, kEnableMenuItem);
		SetMenuItemState(myMenu, IDM_USE_BUFFER, kEnableMenuItem);
		SetMenuItemState(myMenu, IDM_USE_WND_CENTER, kEnableMenuItem);
		SetMenuItemState(myMenu, IDM_COMPOSITE, kEnableMenuItem);
		
		SetMenuItemCheck(myMenu, IDM_USE_BUFFER, (**myAppData).fUseMovieGWorld);
		SetMenuItemCheck(myMenu, IDM_USE_WND_CENTER, !(**myAppData).fUseMovieCenter);
		SetMenuItemCheck(myMenu, IDM_COMPOSITE, (**myAppData).fCompositeMovie);
		
	} else {
		SetMenuItemState(myMenu, IDM_LOAD_MOVIE, kDisableMenuItem);
		SetMenuItemState(myMenu, IDM_SET_CENTER, kDisableMenuItem);
		SetMenuItemState(myMenu, IDM_SET_WIDTH, kDisableMenuItem);
		SetMenuItemState(myMenu, IDM_SET_SCALE, kDisableMenuItem);
		SetMenuItemState(myMenu, IDM_SET_RECTANGLE, kDisableMenuItem);
		SetMenuItemState(myMenu, IDM_SET_CHROMA, kDisableMenuItem);
		SetMenuItemState(myMenu, IDM_SPIN_MOVIE, kDisableMenuItem);
		SetMenuItemState(myMenu, IDM_USE_BUFFER, kDisableMenuItem);
		SetMenuItemState(myMenu, IDM_USE_WND_CENTER, kDisableMenuItem);
		SetMenuItemState(myMenu, IDM_COMPOSITE, kDisableMenuItem);
	}
}


//////////
//
// DoApplicationEventLoopAction
// Perform any application-specific event loop actions.
//
// Return true to indicate that we've completely handled the event here, false otherwise.
//
//////////

Boolean DoApplicationEventLoopAction (EventRecord *theEvent)
{
	return(false);			// no-op for now
}


//////////
//
// AddControllerFunctionality
// Configure the movie controller.
//
//////////

void AddControllerFunctionality (MovieController theMC)
{
	long			myControllerFlags;
	
	// CLUT table use	
	MCDoAction(theMC, mcActionGetFlags, &myControllerFlags);
	MCDoAction(theMC, mcActionSetFlags, (void *)(myControllerFlags | mcFlagsUseWindowPalette));

	// enable keyboard event handling	
	MCDoAction(theMC, mcActionSetKeysEnabled, (void *)true);
	
	// disable drag support
	MCDoAction(theMC, mcActionSetDragEnabled, (void *)false);
}


//////////
//
// InitApplicationWindowObject
// Do any application-specific initialization of the window object.
//
//////////

void InitApplicationWindowObject (WindowObject theWindowObject)
{
	Track					myQTVRTrack = NULL;
	Movie					myMovie = NULL;
	MovieController			myMC = NULL;
	QTVRInstance			myInstance = NULL;
	ApplicationDataHdl		myAppData;
		
	if (theWindowObject == NULL)
		return;

	// make sure we can safely call the QTVR API
	if (!gQTVRMgrIsPresent)
		return;

	// find the QTVR track, if there is one
	myMC = (**theWindowObject).fController;
	myMovie = (**theWindowObject).fMovie;
	myQTVRTrack = QTVRGetQTVRTrack(myMovie, 1);
	
	QTVRGetQTVRInstance(&myInstance, myQTVRTrack, myMC);
	(**theWindowObject).fInstance = myInstance;

	// do any QTVR window configuration
	if (myInstance != NULL) {
		
		// set unit to radians
		QTVRSetAngularUnits(myInstance, kQTVRRadians);

		// do QuickTime window configuration
		myAppData = VRMoov_InitWindowData(theWindowObject);
		(**theWindowObject).fAppData = (Handle)myAppData;
		
	}
}


//////////
//
// RemoveApplicationWindowObject
// Do any application-specific clean-up of the window object.
//
//////////

void RemoveApplicationWindowObject (WindowObject theWindowObject)
{
	OSErr				myErr = noErr;
	QTVRInstance		myInstance = NULL;
	ApplicationDataHdl	myAppData;
		
	// application clean-up
	myAppData = (ApplicationDataHdl)GetAppDataFromWindowObject(theWindowObject);
	if (myAppData != NULL) {
				
		DisposeHandle((Handle)myAppData);
		(**theWindowObject).fAppData = NULL;
	}

	// DoDestroyMovieWindow in MacFramework.c or MovieWndProc in WinFramework.c
	// releases the window object itself
}


//////////
//
// ApplicationMCActionFilterProc 
// Intercept some mc actions for the movie controller.
//
// NOTE: The theRefCon parameter is a handle to a window object record.
//
//////////

PASCAL_RTN Boolean ApplicationMCActionFilterProc (MovieController theMC, short theAction, void *theParams, long theRefCon)
{
#pragma unused(theMC, theParams)

	Boolean				isHandled = false;
	WindowObject		myWindowObject = NULL;
	
	myWindowObject = (WindowObject)theRefCon;
	if (myWindowObject == NULL)
		return(isHandled);
		
	switch (theAction) {
	
		// handle window resizing
		case mcActionControllerSizeChanged:
			SizeWindowToMovie(myWindowObject);
			break;

		// handle idle events
		case mcActionIdle:
			DoIdle((**myWindowObject).fWindow);
			break;
			
		default:
			break;
			
	}	// switch (theAction)
	
	return(isHandled);	
}